/*
 * Detect operating system and browser and embed Autograph player object or 
 * fallback to screenshot with link for mac or youtube video or screenshot.
 * Author: @simon_woodhead
 *
 * Acknowledgements: Includes customised versions of BrowserDetect by PPK
 * and PluginDetect by Eric Gerds.
 *
 * Examples:
 *
 * 1. Set options for individual elements using HTML5 data attributes.
 *
 * <div class="foo"
 *     data-filename="foo.agg"
 *     data-pagetype="2D"
 *     data-screenshot="foo.png"
 *     data-width="640px"
 *     data-height="480px">
 * </div>
 *
 * $('.foo').agg();
 *
 * 2. Set options for all matched elements by passing arguments to agg().
 *
 * $('.foo').agg({ showtoolbar: 0 });
 * 
 * 3. Set global defaults values using the Agg.defaults object.
 *
 * Agg.defaults.isMessageShown = false
 *
 */

;(function( $, window, document, undefined ){
	
	'use strict';
	var pluginName = 'agg',
		Plugin = function( element, options ){
			this.element = element;
			this.$element = $(element);
			this.options = options;
            this.metadata = this.$element.data();
		            
			this._name = pluginName;
		},                
        // parseURI and absolutizeURI from Yaffle
        // https://gist.github.com/1088850
        parseURI = function( url ) {
          var m = String(url).replace(/^\s+|\s+$/g, '')
            .match(/^([^:\/?#]+:)?(\/\/(?:[^:@]*(?::[^:@]*)?@)?(([^:\/?#]*)(?::(\d*))?))?([^?#]*)(\?[^#]*)?(#[\s\S]*)?/);
          // authority = '//' + user + ':' + pass '@' + hostname + ':' port
          return (m ? {
            href     : m[0] || '',
            protocol : m[1] || '',
            authority: m[2] || '',
            host     : m[3] || '',
            hostname : m[4] || '',
            port     : m[5] || '',
            pathname : m[6] || '',
            search   : m[7] || '',
            hash     : m[8] || ''
          } : null);
        },

        absolutizeURI = function( base, href ) {// RFC 3986

          var removeDotSegments = function( input ) {
            var output = [];
            input.replace(/^(\.\.?(\/|$))+/, '')
                 .replace(/\/(\.(\/|$))+/g, '/')
                 .replace(/\/\.\.$/, '/../')
                 .replace(/\/?[^\/]*/g, function (p) {
              if (p === '/..') {
                output.pop();
              } else {
                output.push(p);
              }
            });
            return output.join('')
                .replace(/^\//, input.charAt(0) === '/' ? '/' : '');
          }

          href = parseURI(href || '');
          base = parseURI(base || '');

          return !href || !base ? null : (href.protocol || base.protocol) +
                 (href.protocol || href.authority ? href.authority : 
                 base.authority) + removeDotSegments(href.protocol || 
                 href.authority || href.pathname.charAt(0) === '/' ? 
                 href.pathname : (href.pathname ? ((base.authority && 
                 !base.pathname ? '/' : '') + base.pathname.slice(0, 
                 base.pathname.lastIndexOf('/') + 1) + href.pathname) :
                 base.pathname)) + (href.protocol || href.authority || 
                 href.pathname ? href.search : (href.search || 
                 base.search)) + href.hash;
        };

	Plugin.prototype = {
		defaults: {
            // Autograph player settings
			filename: '',
			pagetype: '2D',
			showtoolbar: '1',
			width: '640px',
			height: '480px',

            // Minimum required version of the player
            minVersion: '3,5,0,6',
			
            // Fallback settings
			screenshot: '',
			hoverMac: 'http://www.autograph-maths.com/activities/' +
				'img/screenshots/open-mac-player.png',
			isOpenWholePage: false,
            popupWidth: '1024px',
            popupHeight: '608px',
            hoverMobile: 'http://www.autograph-maths.com/activities/' +
				'img/screenshots/unsupported-device.png',
			youtube: '',

            // Messages in the event that the player cannot be displayed
			isMessageShown: 'true',
            messageClass: 'message',
			messageContainer: '#autograph-plugin-messages',
			messages: {
				mobile: '<p>Unfortunately this device does not support the ' +
					'Autograph Player. Please view this web page on a ' +
					'Windows or Mac computer.</p>',
				mac: '<p>Click on the screenshot to open the Autograph file ' +
					'in the Autograph Mac Player.</p>',
				windows: {
					ie: {
						x64: '<p>You are using the 64-bit version of ' +
							'Internet Explorer, which does not currently ' +
							'support the Autograph Player. Please open this ' +
							'page in a different browser, e.g. the 32-bit ' +
							'version of Internet Explorer.</p>',
						activeXDisabled: '<p>ActiveX is disabled.<br><br>To ' +
							'enable ActiveX go to Tools &rarr; Internet ' +
							'Options &rarr; Security. Either select a ' +
							'security level that allows ActiveX or fine-tune ' +
							'by clicking Custom level.</p>',
						notInstalled: '<p>The Autograph Player is not ' +
							'installed.<br><br>Autograph Player 3.5.0.6 ' +
							'is required. If present follow the automatic ' +
							'installation procedure, otherwise go to the ' +
							'<a href=\"http://www.autograph-maths.com/' +
							'activities/help.html\" target=\"_blank\">help ' +
							'page</a> for instructions on how to install.</p>',
						oldVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>',
						noVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>'
					},
					firefox: {
						notInstalled: '<p>The Autograph Player is not ' +
							'installed.<br><br>Autograph Player 3.5.0.6 ' +
							'is required. If present click on the "Install ' +
							'Missing Plugins..." button, otherwise go to ' +
							'the <a href=\"http://www.autograph-maths.com/' +
							'activities/help.html\" target=\"_blank\">help ' +
							'page</a> for instructions on how to install.</p>',
						oldVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>',
						noVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>'
					},
					chrome: {
						notInstalled: '<p>The Autograph Player is not ' +
							'installed.<br><br>Autograph Player 3.5.0.6 ' +
							'is required please go to the <a href=\"http://' +
							'www.autograph-maths.com/activities/help.html\"' +
							' target=\"_blank\">help page</a>  for ' +
							'instructions on how to install.</p>',
						oldVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>',
						noVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>'
					},
					other: {
						notInstalled: '<p>The Autograph Player is not ' +
							'installed.<br><br>Autograph Player 3.5.0.6 ' +
							'is required please go to the <a href=\"http://' +
							'www.autograph-maths.com/activities/help.html\"' +
							' target=\"_blank\">help page</a> for ' +
							'instructions on how to install.</p>',
						oldVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>',
						noVersion: '<p>An old version of the Autograph ' +
							'Player is installed.<br><br>Autograph Player ' +
							'3.5.0.6 is required. If present follow the ' +
							'automatic installation procedure, otherwise go ' +
							'to the <a href=\"http://www.autograph-maths.' +
							'com/activities/help.html\" target=\"_blank\">' +
							'help page</a> for instructions on how to ' +
							'install.</p>'
					}
					
				}
				
			}
		},

		init: function() {
		
			this.config = $.extend(
				{},
				this.defaults,
				this.options,
				this.metadata);

			this.browserDetect = this._browserDetect();
            
			this.pluginDetect = this._pluginDetect( this.config.minVersion );
                        
			if ( this.browserDetect.os === 'Windows' ){

				if ( window.navigator.cpuClass === "x64" ) {
				
					// Internet Explorer 64 bit (player not supported)
					this._showMessage(this.config.messages.windows.ie.x64);
				
				} else {
				
					switch ( this.pluginDetect.isMinVersion ) {
						
					case -2:
						// ActiveX is disabled
						this._showMessage(
							this.config.messages.windows.ie.activeXDisabled
						);
						break;
						
					case -1.5:
						// Not installed
						if ( this.browserDetect.browser === 'Explorer' ) {
							this._showMessage(
								this.config.messages.windows.ie.notInstalled
							);
						} else if ( this.browserDetect.browser === 'Firefox' ) {
							this._showMessage(
								this.config.messages.windows.firefox.notInstalled
							);
						} else if ( this.browserDetect.browser === 'Chrome' ) {
							this._showMessage(
								this.config.messages.windows.chrome.notInstalled
							);
						} else {
							this._showMessage(
								this.config.messages.windows.other.notInstalled
							);
						}
						break;
						
					case -0.1:
						// Old version
						if ( this.browserDetect.browser === 'Explorer' ) {
							this._showMessage(
								this.config.messages.windows.ie.oldVersion
							);
						} else if ( this.browserDetect.browser === 'Firefox' ) {
							this._showMessage(
								this.config.messages.windows.firefox.oldVersion
							);
						} else if ( this.browserDetect.browser === 'Chrome' ) {
							this._showMessage(
								this.config.messages.windows.chrome.oldVersion
							);
						} else {
							this._showMessage(
								this.config.messages.windows.other.oldVersion
							);
						}
						break;
						
					case 0:
						// Installed but no version number
						// (so actually < 3.5.0.6)
						if ( this.browserDetect.browser === 'Explorer' ) {
							this._showMessage(
								this.config.messages.windows.ie.noVersion
							);
						} else if ( this.browserDetect.browser === 'Firefox' ) {
							this._showMessage(
								this.config.messages.windows.firefox.noVersion
							);
						} else if ( this.browserDetect.browser === 'Chrome' ) {
							this._showMessage(
								this.config.messages.windows.chrome.noVersion
							);
						} else {
							this._showMessage(
								this.config.messages.windows.other.noVersion
							);
						}
						break;
						
					case 1:
						// Installed and newer than 3.5.0.6
						// No action required
						break;
						
					default:
						// If -3 then a bad argument has been supplied to
						// isMinVersion()
						break;
										
					}
					
					this._embed();
											
				}
				
			} else if ( this.browserDetect.os === 'Mac' ) {

				if ( this.browserDetect.browser === 'AutographMacPlayer' ) {
					
					this._embed();
					
				} else {
				
					// If you are in any other browser provide a link to open
					// the agg file in the Autograph Player
					
					this._showMessage(
						this.config.messages.mac
					);
					
					this._macFallback();
					
				}
					
			} else {

				// Linux, tablet or mobile
				this._showMessage(
					this.config.messages.mobile
				);

				this._mobileFallback();

			}
			
			return this;
		},
		
		_browserDetect: function() {

			/*
			BrowserDetect by PPK
			http://www.quirksmode.org/js/detect.html

			Editted to return the correct result when using the Autograph
			Mac Player by @simon_woodhead
			*/
			var BrowserDetect = {
				init: function () {
					this.browser = this.searchString(this.dataBrowser) ||
						"An unknown browser";
					this.version = this.searchVersion(navigator.userAgent) ||
						this.searchVersion(navigator.appVersion) ||
						"an unknown version";
					this.OS = this.searchString(this.dataOS) || "an unknown OS";
				},
				searchString: function (data) {
					var i,
						dataString,
						dataProp;
					for (i=0;i<data.length;i = i + 1)	{
						dataString = data[i].string;
						dataProp = data[i].prop;
						this.versionSearchString = data[i].versionSearch ||
							data[i].identity;
						if (dataString) {
							if (dataString.indexOf(data[i].subString) != -1) {
								return data[i].identity;
							}
						} else if (dataProp) {
							return data[i].identity;
						}
					}
				},
				searchVersion: function (dataString) {
					var index = dataString.indexOf(this.versionSearchString);
					if (index == -1) {
						return;
					}
					return parseFloat(dataString.substring(
						index + this.versionSearchString.length + 1));
				},
				dataBrowser: [
					{
						string: navigator.userAgent,
						subString: "AutographPlayer",
						identity: "AutographMacPlayer"
					},
					{
						string: navigator.userAgent,
						subString: "Chrome",
						identity: "Chrome"
					},
					{
						string: navigator.userAgent,
						subString: "OmniWeb",
						versionSearch: "OmniWeb/",
						identity: "OmniWeb"
					},
					{
						string: navigator.vendor,
						subString: "Apple",
						identity: "Safari",
						versionSearch: "Version"
					},
					{
						prop: window.opera,
						identity: "Opera",
						versionSearch: "Version"
					},
					{
						string: navigator.vendor,
						subString: "iCab",
						identity: "iCab"
					},
					{
						string: navigator.vendor,
						subString: "KDE",
						identity: "Konqueror"
					},
					{
						string: navigator.userAgent,
						subString: "Firefox",
						identity: "Firefox"
					},
					{
						string: navigator.vendor,
						subString: "Camino",
						identity: "Camino"
					},
					{		// for newer Netscapes (6+)
						string: navigator.userAgent,
						subString: "Netscape",
						identity: "Netscape"
					},
					{
						string: navigator.userAgent,
						subString: "MSIE",
						identity: "Explorer",
						versionSearch: "MSIE"
					},
					{
						string: navigator.userAgent,
						subString: "Gecko",
						identity: "Mozilla",
						versionSearch: "rv"
					},
					{
						// for older Netscapes (4-)
						string: navigator.userAgent,
						subString: "Mozilla",
						identity: "Netscape",
						versionSearch: "Mozilla"
					}
				],
				dataOS : [
					{
						string: navigator.userAgent,
						subString: "AutographPlayer",
						identity: "Mac"
					},
					{
						string: navigator.platform,
						subString: "Win",
						identity: "Windows"
					},
					{
						string: navigator.platform,
						subString: "Mac",
						identity: "Mac"
					},
					{
						string: navigator.userAgent,
						subString: "iPhone",
						identity: "iPhone/iPod"
					},
					{
						string: navigator.platform,
						subString: "Linux",
						identity: "Linux"
					}
				]

			};
			BrowserDetect.init();

			return { os: BrowserDetect.OS, browser: BrowserDetect.browser };
		},

		_pluginDetect: function( minVersion ) {
			
			/*
			PluginDetect v0.7.8 www.pinlady.net/PluginDetect/
			www.pinlady.net/PluginDetect/license/

			Editted to include custom Autograph detection by @simon_woodhead
			*/
			var PluginDetect = {
				version: "0.7.8",
				name: "PluginDetect",
				handler: function (c, b, a) {
					return function () {
						c(b, a);
					};
				},
				isDefined: function (b) {
					return typeof b != "undefined";
				},
				isArray: function (b) {
					return (/array/i).test(Object.prototype.toString.call(b));
				},
				isFunc: function (b) {
					return typeof b == "function";
				},
				isString: function (b) {
					return typeof b == "string";
				},
				isNum: function (b) {
					return typeof b == "number";
				},
				isStrNum: function (b) {
					return (typeof b == "string" && (/\d/).test(b));
				},
				getNumRegx: /[\d][\d\.\_,-]*/,
				splitNumRegx: /[\.\_,-]/g,
				getNum: function (b, c) {
					var d = this,
						a = d.isStrNum(b) ? (d.isDefined(c) ? new RegExp(c) :
							d.getNumRegx).exec(b) : null;
					return a ? a[0] : null;
				},
				compareNums: function (h, f, d) {
					var e = this,
						c, b, a, g = parseInt;
					if (e.isStrNum(h) && e.isStrNum(f)) {
						if (e.isDefined(d) && d.compareNums) {
							return d.compareNums(h, f);
						}
						c = h.split(e.splitNumRegx);
						b = f.split(e.splitNumRegx);
						for (a = 0;
						a < Math.min(c.length, b.length);
						a = a + 1) {
							if (g(c[a], 10) > g(b[a], 10)) {
								return 1;
							}
							if (g(c[a], 10) < g(b[a], 10)) {
								return -1;
							}
						}
					}
					return 0;
				},
				formatNum: function (b, c) {
					var d = this,
						a, e;
					if (!d.isStrNum(b)) {
						return null;
					}
					if (!d.isNum(c)) {
						c = 4;
					}
					c = c - 1;
					e = b.replace(/\s/g, "").split(d.splitNumRegx)
						.concat(["0", "0", "0", "0"]);
					for (a = 0;
					a < 4;
					a = a + 1) {
						if (/^(0+)(.+)$/.test(e[a])) {
							e[a] = RegExp.$2;
						}
						if (a > c || !(/\d/).test(e[a])) {
							e[a] = "0";
						}
					}
					return e.slice(0, 4).join(",");
				},
				$$hasMimeType: function (a) {
					return function (c) {
						if (!a.isIE && c) {
							var f, e, b, d = a.isArray(c) ? c :
								(a.isString(c) ? [c] : []);
							for (b = 0;
							b < d.length;
							b = b + 1) {
								if (a.isString(d[b]) && /[^\s]/.test(d[b])) {
									f = navigator.mimeTypes[d[b]];
									e = f ? f.enabledPlugin : 0;
									if (e && (e.name || e.description)) {
										return f;
									}
								}
							}
						}
						return null;
					};
				},
				findNavPlugin: function (l, e, c) {
					var j = this,
						h = new RegExp(l, "i"),
						d = (!j.isDefined(e) || e) ? /\d/ : 0,
						k = c ? new RegExp(c, "i") : 0,
						a = navigator.plugins,
						g = "",
						f, b, m;
					for (f = 0;
					f < a.length;
					f = f + 1) {
						m = a[f].description || g;
						b = a[f].name || g;
						if ((h.test(m) && (!d || d.test(RegExp.leftContext +
							RegExp.rightContext))) || (h.test(b) && (!d ||
							d.test(RegExp.leftContext +
							RegExp.rightContext)))) {
							if (!k || !(k.test(m) || k.test(b))) {
								return a[f];
							}
						}
					}
					return null;
				},
				getMimeEnabledPlugin: function (k, m, c) {
					var e = this,
						f, b = new RegExp(m, "i"),
						h = "",
						g = c ? new RegExp(c, "i") : 0,
						a, l, d, j = e.isString(k) ? [k] : k;
					for (d = 0;
					d < j.length;
					d = d + 1) {
						if ((f = e.hasMimeType(j[d])) && (f = f.enabledPlugin)) {
							l = f.description || h;
							a = f.name || h;
							if (b.test(l) || b.test(a)) {
								if (!g || !(g.test(l) || g.test(a))) {
									return f;
								}
							}
						}
					}
					return 0;
				},
				AXO: window.ActiveXObject,
				getAXO: function (a) {
					var f = null,
						d, b = this;
					try {
						f = new b.AXO(a);
					} catch (d) {}
					return f;
				},
				convertFuncs: function (f) {
					var a, g, d, b = /^[\$][\$]/;
					for (a in f) {
						if (b.test(a)) {
							try {
								g = a.slice(2);
								if (g.length > 0 && !f[g]) {
									f[g] = f[a](f);
									delete f[a];
								}
							} catch (d) {}
						}
					}
				},
				initObj: function (e, b, d) {
					var a, c;
					if (e) {
						if (e[b[0]] == 1 || d) {
							for (a = 0;
							a < b.length;
							a = a + 2) {
								e[b[a]] = b[a + 1];
							}
						}
						for (a in e) {
							c = e[a];
							if (c && c[b[0]] == 1) {
								this.initObj(c, b);
							}
						}

					}
				},
				initScript: function () {
					var c = this,
						a = navigator,
						e = "/",
						f, i = a.userAgent || "",
						g = a.vendor || "",
						b = a.platform || "",
						h = a.product || "",
						d, j;
					c.initObj(c, ["$", c]);
					for (f in c.Plugins) {
						if (c.Plugins[f]) {
							c.initObj(c.Plugins[f], ["$", c, "$$", c.Plugins[f]], 1);
						}
					}
					c.OS = 100;
					if (b) {
						d = ["Win", 1, "Mac", 2, "Linux", 3, "FreeBSD", 4,
							"iPhone", 21.1, "iPod", 21.2, "iPad", 21.3,
							"Win.*CE", 22.1, "Win.*Mobile", 22.2,
							"Pocket\\s*PC", 22.3, "", 100];
						for (f = d.length - 2;
						f >= 0;
						f = f - 2) {
							if (d[f] && new RegExp(d[f], "i").test(b)) {
								c.OS = d[f + 1];
								break;
							}
						}
					}
					c.convertFuncs(c);
					c.head = (document.getElementsByTagName("head")[0] ||
						document.getElementsByTagName("body")[0] ||
						document.body || null);
					c.isIE = (new Function("return " + e + "*@cc_on!@*" + e + "false"))();
					c.verIE = c.isIE && (/MSIE\s*(\d+\.?\d*)/i).test(i) ?
						parseFloat(RegExp.$1, 10) : null;
					c.ActiveXEnabled = false;
					if (c.isIE) {
						j = ["Msxml2.XMLHTTP", "Msxml2.DOMDocument",
							"Microsoft.XMLDOM", "ShockwaveFlash.ShockwaveFlash",
							"TDCCtl.TDCCtl", "Shell.UIHelper",
							"Scripting.Dictionary", "wmplayer.ocx"];
						for (f = 0;
						f < j.length;
						f = f + 1) {
							if (c.getAXO(j[f])) {
								c.ActiveXEnabled = true;
								break;
							}
						}
					}
					c.isGecko = (/Gecko/i).test(h) && (/Gecko\s*\/\s*\d/i).test(i);
					c.verGecko = c.isGecko ?
						c.formatNum((/rv\s*\:\s*([\.\,\d]+)/i).test(i) ?
						RegExp.$1 : "0.9") : null;
					c.isChrome = (/Chrome\s*\/\s*(\d[\d\.]*)/i).test(i);
					c.verChrome = c.isChrome ? c.formatNum(RegExp.$1) : null;
					c.isSafari = ((/Apple/i).test(g) || (!g && !c.isChrome)) &&
						(/Safari\s*\/\s*(\d[\d\.]*)/i).test(i);
					c.verSafari = c.isSafari &&
						(/Version\s*\/\s*(\d[\d\.]*)/i).test(i) ?
						c.formatNum(RegExp.$1) : null;
					c.isOpera = (/Opera\s*[\/]?\s*(\d+\.?\d*)/i).test(i);
					c.verOpera = c.isOpera &&
						((/Version\s*\/\s*(\d+\.?\d*)/i).test(i) || 1) ?
						parseFloat(RegExp.$1, 10) : null;
				},
				init: function (d) {
					var c = this,
						b, a = {
							status: -3,
							plugin: 0
						};
					if (!c.isString(d)) {
						return a;
					}
					if (d.length == 1) {
						c.getVersionDelimiter = d;
						return a;
					}
					d = d.toLowerCase().replace(/\s/g, "");
					b = c.Plugins[d];
					if (!b || !b.getVersion) {
						return a;
					}
					a.plugin = b;
					if (!c.isDefined(b.installed)) {
						b.installed = null;
						b.version = null;
						b.version0 = null;
						b.getVersionDone = null;
						b.pluginName = d;
					}
					c.garbage = false;
					if (c.isIE && !c.ActiveXEnabled && d !== "java") {
						a.status = -2;
						return a;
					}
					a.status = 1;
					return a;
				},
				$$isMinVersion: function (a) {
					return function (h, g, d, c) {
						var e = a.init(h),
							f, b = -1;
						if (e.status < 0) {
							return e.status;
						}
						f = e.plugin;
						g = a.formatNum(a.isNum(g) ? g.toString() :
							(a.isStrNum(g) ? a.getNum(g) : "0"));
						if (f.getVersionDone != 1) {
							f.getVersion(g, d, c);
							if (f.getVersionDone === null) {
								f.getVersionDone = 1;
							}
						}
						a.cleanup();
						if (f.installed !== null) {
							b = f.installed <= 0.5 ? f.installed :
								(f.installed == 0.7 ? 1 : (f.version === null ?
								0 : (a.compareNums(f.version, g, f) >= 0 ? 1 :
								-0.1)));
						}
						return b;
					};
				},
				getVersionDelimiter: ",",
				$$getVersion: function (a) {
					return function (g, d, c) {
						var e = a.init(g),
							f, b;
						if (e.status < 0) {
							return null;
						}
						f = e.plugin;
						if (f.getVersionDone != 1) {
							f.getVersion(null, d, c);
							if (f.getVersionDone === null) {
								f.getVersionDone = 1;
							}
						}
						a.cleanup();
						b = (f.version || f.version0);
						b = b ? b.replace(a.splitNumRegx, a.getVersionDelimiter) : b;
						return b;
					};
				},
				cleanup: function () {},
				Plugins: {
					autograph: {
						mimeType: ["application/x-agraph-plugin"],
						progID: "Autograph.AutographCtrl",
						classID: "clsid:2DFFF8F7-027B-46FB-9E24-21434BFCB82D",
						getVersion: function () {
							var h = this,
								f = h.$,
								j, k = null,
								c = null,
								a = null;
							if (!f.isIE) {
								c = f.findNavPlugin("Autograph", 0);
								if (c && c.description && f.hasMimeType(h.mimeType)) {
									k = f.getNum(c.description);
								}
							} else {
								c = f.getAXO(h.progID);
								try {
									a = c.Version;
								} catch (j) {}
								if (f.isString(a) && a.length > 0) {
									k = f.getNum(a);
								}
							}
										h.installed = c && k ? 1 : (c ? 0 : -1.5);
							h.version = f.formatNum(k);
						}
					},
					zz: 0
				}
			};
			PluginDetect.initScript();

            return { isMinVersion: PluginDetect.isMinVersion("Autograph",
				minVersion)};

		},

		_embed: function() {
			$('<object class="agraphaxcontrol" ' +
				'classid="clsid:2dfff8f7-027b-46fb-9e24-21434bfcb82d" ' +
				'codebase="http://www.autograph-maths.com/ieplayer/' +
				'AutographPlayer.cab#version=' + this.config.minVersion + '" ' +
				'width=' + this.config.width + ' ' +
				'height=' + this.config.height +'>' +
				'<param name="File" value="' + this.config.filename + '" />' +
				'<param name="PageType" value="' + this.config.pagetype +
				'" />' +
				'<param name="ShowToolbar" value="' + this.config.showtoolbar +
				'" />' +
				'<embed src="' + this.config.filename +
				'" type="application/x-agraph-plugin" ' +
				'pluginspage="http://www.autograph-maths.com/firefoxplugin/' +
				'AutographFirefoxPlugin.xpi" ' +
				'width=' + this.config.width + ' ' +
				'height=' + this.config.height + ' ' +
				'PageType=' + this.config.pagetype + ' ' +
				'ShowToolbar=' + this.config.showtoolbar +
				'></embed></object>')
			.appendTo(this.element);
		},
		
		_macFallback: function() {
			var self = this,
				href,
				screenshot,
                placeholder,                
                url,                
                appendIframe;
            
            // Bug in Wine requires the Z:/
            href = window.location.href.replace("file:///", "file:///Z:/");

            if ( this.config.isOpenWholePage ) {
                url = 'autograph:url=' + href + ';' +
				    'title=' + encodeURIComponent( document.title ) + ';' +
				    'width=' + self.config.popupWidth + ';' +
				    'height=' + self.config.popupHeight + ';' +
				    'minwidth=' + self.config.width + ';' +
				    'minheight=' + self.config.height + ';' +
				    'toolbar=0;location=0;';
            } else { 
                url = 'autograph:url=' + 'http://autograph-maths.com/' +
				    'activities/open.html?path=' +
                    absolutizeURI( window.location.href, this.config.filename ) +
                    '&' + 'type=' + self.config.pagetype + 
                    '&' + 'showtoolbar=' + self.config.showtoolbar + ';' +
				    'title=Autograph%20Activities;' +
				    'width=' + self.config.width + ';' +
				    'height=' + self.config.height + ';' +
				    'minwidth=' + self.config.width + ';' +
				    'minheight=' + self.config.height + ';' +
				    'toolbar=0;location=0;';
            }

			appendIframe = function() {
				$('<iframe />', {
					src: url,
					style: 'display:none'
				}).appendTo('body');
				return false;
			};
			
            if (this.config.screenshot !== ''){
                screenshot = this.config.screenshot;
            } else {
                screenshot = this.config.hoverMac;
            }

			placeholder = $('<a>', {
				'class': 'ir',
				click: appendIframe,
				css: {
					'background-image': 'url(' + screenshot + ')',
					'background-color': 'black',
					'background-repeat': 'no-repeat',
					'background-position': 'center',
					height: self.config.height,
					width: self.config.width
				},
				href: '#',				
				text: 'Click to Play'
			})
            
            // Only add the hover event if the screenshot and hover images
            // are different.
            if (screenshot !== this.config.hoverMac) {
                placeholder.hover(
                    function() {
					    $(this).css("background-image",
						    "url(" + self.config.hoverMac + ")"
					    );},
				    function() {
					    $(this).css("background-image",
						    "url(" + screenshot + ")"
					    );}
                );
            }
            
            placeholder.appendTo(self.element);
						
		},
		
		_mobileFallback: function(){
			var self = this,
				youtubeUrl,
                screenshot,
				placeholder,
                formatYouTubeUrl = function(url) {
					var regExp = /^.*(?:(?:youtu.be\/)|(?:v\/)|(?:\/u\/\w\/)|(?:embed\/)|(?:watch))(?:\?v=)?([^#\&\?]*).*/,
						videoId = url.match(regExp);
					if (videoId && videoId[1].length == 11) {
						return 'http://www.youtube.com/v/' + videoId[1];
					} else {
						alert('Unrecognised YouTube link format.');
					}
				};
			if (self.config.youtube !== ''){
				
				// YouTube videos can be embedded with iFrames or Object-Embed
				// tags. The iFrame method generates XSS errors in Chrome so
				// using the object-embed method. See:
				// https://developers.google.com/youtube/
                // player_parameters#AS3_Object_Embeds

				youtubeUrl = formatYouTubeUrl(self.config.youtube);

				$('<object width="' + self.config.width + '" ' +
					'height="' + self.config.height + '">' +
					'<param name="movie" ' +
					'value="' + youtubeUrl + '?version=3"></param>' +
					'<param name="allowFullScreen" value="true"></param>' +
					'<param name="allowScriptAccess" value="always"></param>' +
					'<embed src="' + youtubeUrl +
					'?version=3"' +
					'type="application/x-shockwave-flash"' +
					'allowfullscreen="true"' +
					'allowscriptaccess="always"' +
					'width="' + self.config.width + '" ' +
					'height="' + self.config.height + '"></embed>' +
					'</object>')
					.appendTo(self.element);

			} else {

                if (this.config.screenshot !== ''){
                    screenshot = this.config.screenshot;
                } else {
                    screenshot = this.config.hoverMobile;
                }

				placeholder = $('<div>', {
					css: {
						'background-image':  'url(' + screenshot +
							')',
						'background-repeat': 'no-repeat',
						'background-position': 'center',
						'background-color': 'black',
						height: self.config.height,
						width: self.config.width
					}
				});

                // Only add the hover event if the screenshot and hover images
                // are different.
                if (screenshot !== this.config.hoverMobile) {
                    placeholder.hover(
                        function() {
					        $(this).css("background-image",
						        "url(" + self.config.hoverMobile + ")"
					        );},
				        function() {
					        $(this).css("background-image",
						        "url(" + screenshot + ")"
					        );}
                    );
                }

                placeholder.appendTo(self.element);

			}
						
		},
		
		_showMessage: function( message ){
			if ( this.config.isMessageShown ) {
            
                var data = $(this.config.messageContainer).data(this._name);

			    // Only display the message once
			    if ( !data || !data.isMessagePresent ) {

				    $(message)
					    .addClass(this.config.messageClass)
					    .prependTo(this.config.messageContainer)
					    .fadeIn();

				    $(this.config.messageContainer).data(
					    this._name,
					    {
						    isMessagePresent: true
					    }
				    );

			    }

            }

		}
        				
	};

	Plugin.defaults = Plugin.prototype.defaults;

	$.fn[pluginName] = function( options ) {
		return this.each(function() {
			new Plugin(this, options).init();
		});
	};

	//Expose the plugin class so it can be modified
	window.Agg = Plugin;

})( jQuery, window , document );
